package easik.sketch.constraint;

import java.util.ArrayList;

import easik.sketch.path.SketchPath;
import easik.sketch.vertex.EntityNode;

/** 
 * Represents the product constraint. Contains a list of paths 
 * which all share a source.
 * 
 * @author Kevin Green 2006
 * @author Vera Ranieri 2006
 * @version 2006-05-25 Vera Ranieri
 */
public class ProductConstraint extends Constraint {
	
	
	
	/**
	 * Product constraint constructor. Accepts an ArrayList 
	 * that can have any number of paths.  Gives default coordinates.
	 *  
	 * @param inPaths An ArrayList of SketchPaths
	 */
	public ProductConstraint(ArrayList<SketchPath> inPaths) {

		_constraintLabel = "x";
		_isVisible = true;
		_paths = inPaths;
		
		addEdges();
	}

	/**
	 * Product constraint constructor.  Accepts coordinates in constructor
	 * to allow for user-defined visual placement.
	 * 
	 * @param inPaths An ArrayList of SketchPaths
	 * @param x X coordinate of visual on graph
	 * @param y Y coordinate of visual on graph
	 * @param isVisible If the constraint is visible in the graph or not
	 */
	public ProductConstraint(ArrayList<SketchPath> inPaths, int x, int y, boolean isVisible) {
		super("x",x,y,isVisible);
		_paths = inPaths;
		_constraintLabel = "x";
		
		addEdges();
	}
	
	/**
	 * Sets the path array, updates edge list, and updates display
	 * 
	 * @param inPaths The new array of paths
	 */
	public void setPaths(ArrayList<SketchPath> inPaths){
		_paths = inPaths;
		 
		addEdges();
		setVisible(!isVisible());
		setVisible(!isVisible());
	}
	
	
	/**
	 * Method to determine whether a given set of paths could legally create a product constraint.
	 * 
	 * @param paths The paths potentially involved in a product constraint
	 * @return True if a legal path configuration, false otherwise
	 * @since 2006-05-25 Vera Ranieri
	 */
	public static boolean isProductConstraint(ArrayList paths){
		
		//We must have a bunch of injective arrows aimed at the same spot
		EntityNode sharedSource = null;
		int pathsInvolved = 0;

		SketchPath currentPath;
		int size = paths.size();
		// Search through the selection...
		for (int i = 0; i < size; i++) {
				currentPath = (SketchPath) paths.get(i);

				// First source we come across becomes our 'sharedSource'
				if (sharedSource == null) {
					sharedSource = (EntityNode) currentPath.getDomain();
				} 
				else if (sharedSource != currentPath.getDomain()) {
					// If we encounter a source which is not the sharedSource, fail			
					return false;
				}
				if(sharedSource.equals(currentPath.getCoDomain())){
					// In a loop
					return false;
				}

				// Otherwise, we're fine, add the edge
				pathsInvolved++;
		}

		if (pathsInvolved < 2) {
			return false;
		}
		return true;
	}
}
